/*
 * tn_testlib.c
 * File revision 2
 * A set of tests for tcp net.
 * (c) 2001 Jacob Lundberg, jacob@chaos2.org
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


/*
 * 2001.07.16	initial revision
 * 2001.07.25	removing void
 * 		char -> byte
 * 		fixed bug in round trip test
 * 		added GPL notice
 * 2001.08.15	fix uninitd mem segfaults
 */


#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include "tn_testlib.h"
#include "tn_local.h"
#include "tcpnet.h"

/* For TN_BUFFER_SIZE and such, so we can make evil tests. */
#include "tn_internal.h"


bool test_basic(tn_connection *tnc_ln, tn_connection *tnc_cn) {
/*
 * test_basic
 * Perform a simple transmission test.
 */

   /*
    * Make sure that size_ln matches the length of
    * buf_ln, including the terminating '\0'.
    */
   bool ret = false;
   struct timeval timeout;
   int size_ln = 65, size_cn = -1;
   byte   buf_ln[65] = "0123456789 ABCDEFGHIJKLMNOPQRSTUVWXYZ abcdefghijklmnopqrstuvwxyz";
   byte *buf_cn = (byte *)0xdeadbeef;

   /* Print out this test's header. */
   printf("\nRunning test_basic...\n");

   /* Set a reading timeout. */
   timeout.tv_sec  = 1;
   timeout.tv_usec = 0;

   /* Write it out to packet buffer. */
   if(!tn_write(tnc_ln, buf_ln, size_ln)) {
      printf("  warning - tn_write indicates failure\n");
      fflush(stdout);
   }

   /* Write packet buffer out to network. */
   if(!tn_write_flush(tnc_ln)) {
      printf("  warning - tn_write_flush indicates failure\n");
      fflush(stdout);
   }

   /* Read network to packet buffer. */
   if(!tn_scan_read(tnc_cn, &timeout)) {
      printf("  warning - tn_scan_read indicates failure\n");
      fflush(stdout);
   }

   /* Read packet buffer to local buffer. */
   if(!tn_read(tnc_cn, &buf_cn, &size_cn)) {
      printf("  warning - tn_read indicates failure\n");
      fflush(stdout);
   }

   /* Compare the lengths. */
   if(size_ln != size_cn) {
      printf("test_basic fails due to length %i becoming %i\n", size_ln, size_cn);
      printf("  tnc_ln->state = %#010x, tnc_cn->state = %#010x\n", tnc_ln->state, tnc_cn->state);
      fflush(stdout);
   } else {
      /* Compare the buffers. */
      if(strncmp(buf_ln, buf_cn, size_ln)) {
         buf_cn[size_ln - 1] = '\0';
         printf("test_basic fails due to buffer corruption:\n  send: %s\n  recv: %s\n", buf_ln, buf_cn);
         printf("  tnc_ln->state = %#010x, tnc_cn->state = %#010x\n", tnc_ln->state, tnc_cn->state);
         fflush(stdout);
      } else {
         printf("test_basic success\n");
         fflush(stdout);
         ret = true;
      }
   }

   /* Print the read buffer status message. */
   if(buf_cn == (byte *)0xdeadbeef)
      printf("test_basic - the read buffer was not relocated\n");
   else
      free(buf_cn);

   return(ret);

}


bool test_round_trip(tn_connection *tnc_ln, tn_connection *tnc_cn) {
/*
 * test_round_trip
 * Test the packet engine with round-trip packets.
 * This is the test that could be set up to use random data...
 */

   bool ret = false;
   struct timeval timeout;
   int size_ln = 488;
   byte   buf_ln[488] = "Reach into the thoughts of friends,\n"
                        "And find they do not know your name.\n"
                        "Squeeze the teddy bear too tight,\n"
                        "And watch the feathers burst the seams.\n"
                        "Touch the stained glass with your cheek,\n"
                        "And feel its chill upon your blood.\n"
                        "Hold a candle to the night,\n"
                        "And see the darkness bend the flame.\n"
                        "Tear the mask of peace from God,\n"
                        "And hear the roar of souls in hell.\n"
                        "Pluck a rose in name of love,\n"
                        "And watch the petals curl and wilt.\n"
                        "Lean upon the western wind,\n"
                        "And know you are alone.\n"
                        " -- Dru Mims";
   byte *buf_cn = NULL;
   int size_cn;

   /* Print out this test's header. */
   printf("\nRunning test_round_trip...\n");

   /* Set a reading timeout. */
   timeout.tv_sec  = 1;
   timeout.tv_usec = 0;

   /* [1] Write it out to packet buffer. */
   if(!tn_write(tnc_ln, buf_ln, size_ln)) {
      printf("  warning - tn_write [1] indicates failure\n");
      fflush(stdout);
   }

   /* [1] Write packet buffer out to network. */
   if(!tn_write_flush(tnc_ln)) {
      printf("  warning - tn_write_flush [1] indicates failure\n");
      fflush(stdout);
   }

   /* [1] Read network to packet buffer. */
   if(!tn_scan_read(tnc_cn, &timeout)) {
      printf("  warning - tn_scan_read [1] indicates failure\n");
      fflush(stdout);
   }

   /* [1] Read packet buffer to local buffer. */
   if(!tn_read(tnc_cn, &buf_cn, &size_cn)) {
      printf("  warning - tn_read [1] indicates failure\n");
      fflush(stdout);
   }

   /* [2] Write the packet back to the sender. */
   if(!tn_write(tnc_ln, buf_cn, size_cn)) {
      printf("  warning - tn_write [2] indicates failure\n");
      fflush(stdout);
   } else {
      /* Free the buffer we just sent. */
      free(buf_cn);
      buf_cn = NULL;
      size_cn = 0;
   }

   /* [2] Transmit the return packet. */
   if(!tn_write_flush(tnc_ln)) {
      printf("  warning - tn_write_flush [2] indicates failure\n");
      fflush(stdout);
   }

   /* [2] Read network to packet buffer. */
   if(!tn_scan_read(tnc_cn, &timeout)) {
      printf("  warning - tn_scan_read [2] indicates failure\n");
      fflush(stdout);
   }

   /* [2] Read packet buffer to local buffer. */
   if(!tn_read(tnc_cn, &buf_cn, &size_cn)) {
      printf("  warning - tn_read [2] indicates failure\n");
      fflush(stdout);
   }

   /* Compare the lengths. */
   if(size_ln != size_cn) {
      printf("test_round_trip fails due to modified length\n");
      printf("  tnc_ln->state = %#010x, tnc_cn->state = %#010x\n", tnc_ln->state, tnc_cn->state);
      fflush(stdout);
   } else {
      /* Compare the buffers. */
      if(strncmp(buf_ln, buf_cn, size_ln)) {
         printf("test_round_trip fails due to buffer corruption\n");
         printf("  tnc_ln->state = %#010x, tnc_cn->state = %#010x\n", tnc_ln->state, tnc_cn->state);
         fflush(stdout);
      } else {
         printf("test_round_trip success\n");
         fflush(stdout);
         ret = true;
      }
   }

   free(buf_cn);
   return(ret);

}


bool test_coalesce(tn_connection *tnc_ln, tn_connection *tnc_cn) {
/*
 * test_coalesce
 * Test the packet coalescence code.
 */

   bool ret = false;
   struct timeval timeout;
   byte *rbuf1 = NULL, *rbuf2 = NULL, *rbuf3 = NULL;
   int rlen1, rlen2, rlen3;

   /* The first little bufferlet. */
   int len1 = 30;
   byte  buf1[30] = "asdf1jkl; ;lkj2fdsa asdf3jkl;";

   /* Buffer zwei. */
   int len2 = 48;
   byte  buf2[48] = "`1qazxsw23edcvfr45tgbnhy67ujm,ki89ol./;p0-[']=|";

   /* The Big Buffer. */
   int len3 = TN_BUFFER_SIZE + 1;
   byte  buf3[TN_BUFFER_SIZE + 1];
   memset(buf3, 'X', TN_BUFFER_SIZE);
   buf3[TN_BUFFER_SIZE] = '\0';

   /* Print out this test's header. */
   printf("\nRunning test_coalesce...\n");

   /* Set a reading timeout. */
   timeout.tv_sec  = 1;
   timeout.tv_usec = 0;

   /* Write out the data to buffer. */
   if(!(tn_write(tnc_ln, buf1, len1) &&
        tn_write(tnc_ln, buf2, len2) &&
        tn_write(tnc_ln, buf3, len3))) {
      printf("  warning - tn_write indicates failure\n");
      fflush(stdout);
   }

   /* Write packet buffer out to network. */
   if(!tn_write_flush(tnc_ln)) {
      printf("  warning - tn_write_flush indicates failure\n");
      fflush(stdout);
   }

   /* Read network to packet buffer. */
   if(!tn_scan_read(tnc_cn, &timeout)) {
      printf("  warning - tn_scan_read indicates failure\n");
      fflush(stdout);
   }

   /* Read packet buffer to local buffer. */
   if(!(tn_read(tnc_cn, &rbuf1, &rlen1) &&
        tn_read(tnc_cn, &rbuf2, &rlen2) &&
        tn_read(tnc_cn, &rbuf3, &rlen3))) {
      printf("  warning - tn_read indicates failure\n");
      fflush(stdout);
   }

   /* Compare the lengths. */
   if(len1 != rlen1 || len2 != rlen2 || len3 != rlen3) {
      printf("test_coalesce fails due to modified length\n");
      printf("  tnc_ln->state = %#010x, tnc_cn->state = %#010x\n", tnc_ln->state, tnc_cn->state);
      fflush(stdout);
   } else {
      /* Compare the buffers. */
      if(strncmp(buf1, rbuf1, len1) ||
         strncmp(buf2, rbuf2, len2) ||
         strncmp(buf3, rbuf3, len3)) {
         printf("test_coalesce fails due to buffer corruption\n");
         printf("  tnc_ln->state = %#010x, tnc_cn->state = %#010x\n", tnc_ln->state, tnc_cn->state);
         fflush(stdout);
      } else {
         printf("test_coalesce success\n");
         fflush(stdout);
         ret = true;
      }
   }

   free(rbuf1);
   free(rbuf2);
   free(rbuf3);
   return(ret);

}


bool test_splicing(tn_connection *tnc_ln, tn_connection *tnc_cn) {
/*
 * test_splicing
 * Test the packet splicing code.
 */

   int rlen1;
   bool ret = false;
   byte *rbuf1 = NULL;
   struct timeval timeout;

   /* The whole buffer. */
   int jmp1, pos1 = 0, len1 = 320;
   byte  buf1[320] = "!QAZ@WSX#EDC$RFV%TGB^YHN&UJM*IK<(OL>)P:?1qaz2wsx3edc4rfv5tgb6yhn7ujm8ik,9ol.0p;/"
                     "!@QWASZX#$ERDFCV%^TYGHBN&*UIJKM<()OPL:>?12qwaszx34erdfcv56tyghbn78uijkm,90opl;./"
                     "!QWAZX@#ESDC$RTFVB%^YGHN&UIJM<*(OKL>P:)?1qwazx23esdc4rtfvb56yghn7uijm,89okl.p;0/"
                     "!@#$REWQFDSAZXCV%^&*IUYTKJHGBNM<()PO:L>?1234rewqfdsazxcv5678iuytkjhgbnm,90po;l./";

   /* Print out this test's header. */
   printf("\nRunning test_splicing...\n");

   /* Set a reading timeout. */
   timeout.tv_sec  = 1;
   timeout.tv_usec = 0;

   /* Send the buffer header, one dword at a time... >:) */
   /* The Beginning Of Packet marker. */
   jmp1 = htonl(TN_ALIGN_MAGIC);
   if(!tn_send_buffer(tnc_ln, (byte *)&jmp1, sizeof(dword))) {
      printf("  warning - tn_send_buffer unable to transmit\n");
      fflush(stdout);
   }
   if(!tn_scan_read(tnc_cn, &timeout) && !TN_CONNECTION_IS_UP(tnc_cn)) {
      printf("  warning - tn_scan_read buffer connection is down\n");
      fflush(stdout);
   }
   /* The proper length of this packet. */
   jmp1 = htonl(len1);
   if(!tn_send_buffer(tnc_ln, (byte *)&jmp1, sizeof(dword))) {
      printf("  warning - tn_send_buffer unable to transmit\n");
      fflush(stdout);
   }
   if(!tn_scan_read(tnc_cn, &timeout) && !TN_CONNECTION_IS_UP(tnc_cn)) {
      printf("  warning - tn_scan_read buffer connection is down\n");
      fflush(stdout);
   }
   /* The identification number of this packet. */
   jmp1 = htonl(++(tnc_ln->out_id));
   if(!tn_send_buffer(tnc_ln, (byte *)&jmp1, sizeof(dword))) {
      printf("  warning - tn_send_buffer unable to transmit\n");
      fflush(stdout);
   }
   if(!tn_scan_read(tnc_cn, &timeout) && !TN_CONNECTION_IS_UP(tnc_cn)) {
      printf("  warning - tn_scan_read buffer connection is down\n");
      fflush(stdout);
   }

   /* Send the buffer in pieces. */
   while(len1 - pos1 > 0) {
      /* Pick some arbitrary amount of data to send. */
      jmp1 = pos1 % 17 + 1;
      if(pos1 + jmp1 > len1)
         jmp1 = len1 - pos1;

      /* Write a fragment of the buffer out. */
      if(!tn_send_buffer(tnc_ln, buf1 + pos1, jmp1)) {
         printf("  warning - tn_send_buffer unable to transmit\n");
         fflush(stdout);
         break;
      }

      /* Read network fragment to packet buffer. */
      if(!tn_scan_read(tnc_cn, &timeout) && !TN_CONNECTION_IS_UP(tnc_cn)) {
         printf("  warning - tn_scan_read buffer connection is down\n");
         fflush(stdout);
         break;
      }

      /* Advance forward. */
      pos1 += jmp1;
   }

   /* Read packet buffer to local buffer. */
   if(!tn_read(tnc_cn, &rbuf1, &rlen1)) {
      printf("  warning - tn_read indicates failure\n");
      fflush(stdout);
   }

   /* Compare the lengths. */
   if(len1 != rlen1) {
      printf("test_splicing fails due to modified length\n");
      printf("  tnc_ln->state = %#010x, tnc_cn->state = %#010x\n", tnc_ln->state, tnc_cn->state);
      fflush(stdout);
   } else {
      /* Compare the buffers. */
      if(strncmp(buf1, rbuf1, len1)) {
         printf("test_splicing fails due to buffer corruption\n");
         printf("  tnc_ln->state = %#010x, tnc_cn->state = %#010x\n", tnc_ln->state, tnc_cn->state);
         fflush(stdout);
      } else {
         printf("test_splicing success\n");
         fflush(stdout);
         ret = true;
      }
   }

   free(rbuf1);
   return(ret);

}


bool test_buffering(tn_connection *tnc_ln, tn_connection *tnc_cn) {
/*
 * test_buffering
 * Test for proper buffer separation and ordering constraints.
 */

   bool ret = false;
   struct timeval timeout;
   int size_ln = 44;
   byte   buf_ln[44] = "All work and no play makes Jack a dull boy.";
   byte *buf_cn, *buf_tmp;
   int size_cn;

   /* Print out this test's header. */
   printf("\nRunning test_buffering...\n");

   /* Set a reading timeout. */
   timeout.tv_sec  = 1;
   timeout.tv_usec = 0;

   /* Make a temp output buffer. */
   buf_tmp = (byte *)malloc(size_ln * sizeof(byte));
   if(buf_tmp == NULL) {
      printf("  error - unable to allocate memory; aborting test_buffering\n");
      return(false);
   }
   strncpy(buf_tmp, buf_ln, size_ln);

   /* Write it out to packet buffer. */
   if(!tn_write(tnc_ln, buf_tmp, size_ln)) {
      printf("  warning - tn_write indicates failure\n");
      fflush(stdout);
   }

   /* Munch the buffer into little bits! */
   memset(buf_tmp, '.', size_ln);
   free(buf_tmp);

   /* Write packet buffer out to network. */
   if(!tn_write_flush(tnc_ln)) {
      printf("  warning - tn_write_flush indicates failure\n");
      fflush(stdout);
   }

   /* Read network to packet buffer. */
   if(!tn_scan_read(tnc_cn, &timeout)) {
      printf("  warning - tn_scan_read indicates failure\n");
      fflush(stdout);
   }

   /* Make a temp input buffer. */
   buf_tmp = (byte *)malloc(size_ln * sizeof(byte));
   if(buf_tmp == NULL) {
      printf("  error - unable to allocate memory; aborting test_buffering\n");
      tn_scan_read(tnc_cn, &timeout);
      tn_read(tnc_cn, &buf_cn, &size_cn);
      return(false);
   }
   memset(buf_tmp, '.', size_ln);
   buf_cn = buf_tmp;

   /* Read packet buffer to local buffer. */
   if(!tn_read(tnc_cn, &buf_cn, &size_cn)) {
      printf("  warning - tn_read indicates failure\n");
      fflush(stdout);
   }

   /* Trash the buffer we passed the input function. */
   memset(buf_tmp, '.', size_ln);
   free(buf_tmp);

   /* Compare the lengths. */
   if(size_ln != size_cn) {
      printf("test_buffering fails due to modified length\n");
      printf("  tnc_ln->state = %#010x, tnc_cn->state = %#010x\n", tnc_ln->state, tnc_cn->state);
      fflush(stdout);
   } else {
      /* Compare the buffers. */
      if(strncmp(buf_ln, buf_cn, size_ln)) {
         printf("test_buffering fails due to buffer corruption\n");
         printf("  tnc_ln->state = %#010x, tnc_cn->state = %#010x\n", tnc_ln->state, tnc_cn->state);
         fflush(stdout);
      } else {
         printf("test_buffering success\n");
         fflush(stdout);
         ret = true;
      }
   }

   if(buf_cn != buf_tmp) free(buf_cn);
   return(ret);

}


bool run_tests(tn_connection *tnc_ln, tn_connection *tnc_cn) {
/*
 * run_tests
 * Run the various tcp net tests above.
 */

   bool success = true;

   /* Perform the basic packet transmission test. */
   if(!test_basic(tnc_ln, tnc_cn)) success = false;

   /* Test packet round trips. */
   if(!test_round_trip(tnc_ln, tnc_cn)) success = false;

   /* Test the packet coalescence code. */
   if(!test_coalesce(tnc_ln, tnc_cn)) success = false;

   /* Test the packet splicing. */
   if(!test_splicing(tnc_ln, tnc_cn)) success = false;

   /* Test homicidal buffering. */
   if(!test_buffering(tnc_ln, tnc_cn)) success = false;

   /* Summarize. */
   if(success)
      printf("\nAll tests succeeded.\n");
   else
      printf("\nSome tests failed.\n");
   fflush(stdout);

   return(success);

}
