/*
 * tn_test.c
 * File revision 2
 * A small testing system for tcp net.
 * (c) 2001 Jacob Lundberg, jacob@chaos2.org
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


/*
 * 2001.07.15	initial revision
 * 2001.07.25	added GPL notice
 * 2001.09.09	close sockets even if error
 */


#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include "tn_testlib.h"
#include "tn_local.h"
#include "tcpnet.h"


/* The test program may modify the connection state. */
#include "tn_internal.h"


/* Parameters. */
#define  TN_TEST_PROTOCOL       0
#define  TN_TEST_ADDR           "127.0.0.1"
#define  TN_TEST_LN_PORT        2416
#define  TN_TEST_CN_PORT        6142
#define  TN_TEST_BACKLOG        5


void fatal_error(const char *error, const char *message, int sock_ln, int sock_cn) {
/*
 * fatal_error
 * Something Earth-shaking (well, ok, but bad enough to stop the test) happened.
 */

   /* Tell the user what happened. */
   if(message != NULL)
      printf("\nFatal: %s says '%s'.\nEntering cleanup...\n", error, message);
   else
      printf("\nFatal: %s.\nEntering cleanup...\n", error);
   fflush(stdout);

   /* Stop the sockets, if they were started. */
   if(sock_ln >= 0) {
      shutdown(sock_ln, SHUT_RDWR);
      close(sock_ln);
   }
   if(sock_cn >= 0) {
      shutdown(sock_cn, SHUT_RDWR);
      close(sock_cn);
   }

   /* Terminate now! */
   exit(-1);

}


bool setup_socks(int *socket_ln, int *socket_cn) {
/*
 * setup_socks
 * Set up the sockets for testing.
 */

   int sock_sr, sock_ln, sock_cn;
   struct sockaddr_in socket_info;
   struct in_addr address;

   /* Get some sockets. */
   sock_sr = socket(PF_INET, SOCK_STREAM, TN_TEST_PROTOCOL);
   sock_cn = socket(PF_INET, SOCK_STREAM, TN_TEST_PROTOCOL);
   if(sock_sr < 0 || sock_cn < 0)
      fatal_error("assign sockets", strerror(errno), sock_sr, sock_cn);

   /* General setup for bind. */
   memset((char *)&socket_info, '\0', sizeof(socket_info));
   socket_info.sin_addr.s_addr = INADDR_ANY;
   socket_info.sin_family = AF_INET;

   /* Setup for binding sock_sr. */
   socket_info.sin_port = htons(TN_TEST_LN_PORT);

   /* Try to bind sock_sr. */
   if(bind(sock_sr, (struct sockaddr *)&socket_info, sizeof(socket_info)) < 0)
      fatal_error("bind sock_sr", strerror(errno), sock_sr, sock_cn);

   /* Setup for binding sock_cn. */
   socket_info.sin_port = htons(TN_TEST_CN_PORT);

   /* Try to bind sock_cn. */
   if(bind(sock_cn, (struct sockaddr *)&socket_info, sizeof(socket_info)) < 0)
      fatal_error("bind sock_cn", strerror(errno), sock_sr, sock_cn);

   /* Set sock_sr nonblocking. */
   if(fcntl(sock_sr, F_SETFL, O_NONBLOCK) != 0)
      fatal_error("set sock_sr nonblocking", strerror(errno), sock_sr, sock_cn);

   /* Start listening on sock_sr. */
   if(listen(sock_sr, TN_TEST_BACKLOG) < 0)
      fatal_error("listen sock_sr", strerror(errno), sock_sr, sock_cn);

   /* Setup for connecting sock_cn to sock_sr. */
   if(!inet_aton(TN_TEST_ADDR, &address))
      fatal_error("inet_aton", "invalid test destination", sock_sr, sock_cn);
   socket_info.sin_addr.s_addr = address.s_addr;
   socket_info.sin_port = htons(TN_TEST_LN_PORT);

   /* Attempt a connect to sock_sr via sock_cn. */
   if(connect(sock_cn, (const struct sockaddr *)&socket_info, sizeof(struct sockaddr_in)) < 0)
      fatal_error("connect sock_cn", strerror(errno), sock_sr, sock_cn);

   /* Accept the client connection. */
   sock_ln = accept(sock_sr, NULL, NULL);
   if(sock_ln < 0)
      fatal_error("accept sock_cn from sock_sr", strerror(errno), sock_sr, sock_cn);

   /* Shutdown the listening socket. */
   shutdown(sock_sr, SHUT_RDWR);
   close(sock_sr);

   /* Set sock_ln nonblocking. */
   if(fcntl(sock_ln, F_SETFL, O_NONBLOCK) != 0)
      fatal_error("set sock_ln nonblocking", strerror(errno), sock_ln, sock_cn);

   /* Set sock_cn nonblocking. */
   if(fcntl(sock_cn, F_SETFL, O_NONBLOCK) != 0)
      fatal_error("set sock_cn nonblocking", strerror(errno), sock_ln, sock_cn);

   /* Set the final values. */
   *socket_ln = sock_ln;
   *socket_cn = sock_cn;

   return(true);

}


bool setup_tcpnet(tn_connection **tnc_ln, tn_connection **tnc_cn, int sock_ln, int sock_cn) {
/*
 * setup_tcpnet
 * Set up the tcp net structures.
 */

   /* Set up the sock_ln connection. */
   errno = 0;
   if(!tn_instantiate(tnc_ln, sock_ln)) {
      if(errno)
         fatal_error("tn_instantiate for sock_ln", strerror(errno), sock_ln, sock_cn);
      else
         fatal_error("tn_instantiate failed for sock_ln", NULL, sock_ln, sock_cn);
   }

   /* Set up the sock_cn connection. */
   errno = 0;
   if(!tn_instantiate(tnc_cn, sock_cn)) {
      if(errno)
         fatal_error("tn_instantiate for sock_cn", strerror(errno), sock_ln, sock_cn);
      else
         fatal_error("tn_instantiate failed for sock_cn", NULL, sock_ln, sock_cn);
   }

   return(true);

}


bool _shutdown_single(tn_connection **tnc) {
/*
 * _shutdown_single
 * Shut down a single tn_connection, allowing for error state.
 */

   bool ret = true;

   /* Be very aggressive about the socket shutdown. */
   if((*tnc)->socket >= 0) {
      tn_clr_state((*tnc), TN_STATE_CONNECTED);
      shutdown((*tnc)->socket, SHUT_RDWR);
      close((*tnc)->socket);
      (*tnc)->socket = -1;
      tn_set_state((*tnc), TN_STATE_ERR_CONNLOST);
   } else {
      ret = false;
   }

   /* Run the standard termination code -- which won't do much. */
   if(!tn_terminate(tnc))
      ret = false;

   return(ret);

}


bool shutdown_tcpnet(tn_connection **tnc_ln, tn_connection **tnc_cn) {
/*
 * shutdown_tcpnet
 * Shut down the tcp net system and the sockets.
 */

   /* Close out the tnc_ln connection. */
   errno = 0;
   if(!_shutdown_single(tnc_ln)) {
      if(errno)
         fatal_error("shutdown tnc_ln", strerror(errno), -1, -1);
      else
         fatal_error("shutdown tnc_ln failed", NULL, -1, -1);
   }

   /* Close out the tnc_cn connection. */
   errno = 0;
   if(!_shutdown_single(tnc_cn)) {
      if(errno)
         fatal_error("shutdown tnc_cn", strerror(errno), -1, -1);
      else
         fatal_error("shutdown tnc_cn failed", NULL, -1, -1);
   }

   return(true);

}


int main(int argc, char **argv) {
/*
 * tn_test:main
 * Perform some basic tests on the tcp net suite.
 */

   tn_connection *tnc_ln, *tnc_cn;
   int sock_ln, sock_cn;
   int ret = 0;

   /* Let the user know what they're running. */
   printf("Starting tn_test for tcp net v" VERSION ".\n"
          "(c) 2001, Jacob Lundberg, jacob(at)chaos2.org\n");

   /* Setup for the tests. */
   if(!setup_socks(&sock_ln, &sock_cn))
      fatal_error("setup socks failed", NULL, sock_ln, sock_cn);

   /* Setup the tcp net structs. */
   if(!setup_tcpnet(&tnc_ln, &tnc_cn, sock_ln, sock_cn))
      fatal_error("setup tcpnet failed", NULL, sock_ln, sock_cn);

   /* Perform the tests (decl in tn_testlib). */
   if(!run_tests(tnc_ln, tnc_cn))
      ret = -1;

   if(!shutdown_tcpnet(&tnc_ln, &tnc_cn))
      fatal_error("shutdown tcpnet failed", NULL, sock_ln, sock_cn);

   /* Success (?) */
   return(ret);

}
